﻿/* (c) 2021 Lighting and Imaging SCHOTT North America Inc
 *  This source code is provided as-is with no warranty or guarantee of any kind.
 *  Use at your own risk.
 * 
 *  This program is intended to demonstrate basic remote control of the MC-LS.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Threading;
using System.IO.Ports;

namespace MC_LS_Demo
{
    public partial class MainWindow : Window
    {
        SerialPort serialPort = null;
        DispatcherTimer dispatcherTimer = new DispatcherTimer();

        public MainWindow()
        {
            InitializeComponent();
            InitializeForm();
            
            // Initialize and start the system timer, which polls the status of the MC-LS.
            InitializeTimer(200);
        }

        #region Status Update Functions
        private void InitializeTimer(int timeout_ms)
        {
            dispatcherTimer.Tick += DispatcherTimer_Tick;
            dispatcherTimer.Interval = new TimeSpan(0,0,0,0,timeout_ms);
        }

        private void DispatcherTimer_Tick(object sender, EventArgs e)
        {
            if (null != serialPort)
            {
                try
                {
                    // get current status of MC-LS and update various form controls
                    Status_Update();
                }
                catch (Exception ex)
                {
                    if (ex.Message.Equals("The port is closed."))
                    {
                        SerialPort_Disconnect();
                        MessageBox.Show("The MC-LS was disconnected.");
                    }
                }
            }
        }

        /// <summary>
        /// Gets the status summary from the MC-LS by sending the &xs command and parsing the response.
        /// </summary>
        private void Status_Update()
        {
            string statusString = SendCommand("&xs");
            if ( !string.IsNullOrEmpty(statusString) )
            {
                string[] statusArray = statusString.Split(',');
                /**  &xs command format string for the printf function used in the MC-LS firmware to construct the response:
                 *   &xs%02x,%02x,%03x,%1x,%+05.01f,%+05.01f,%04d,%05.02f,%04d,%04d,%1x,%1x,%1x
                 *   Index   Format  Descripton
                 *   0       %02x        Faults
                 *   1       %02x        Warnings
                 *   2       %03x        Led Intensity as a hex value between 0 and 2047 (0 - 7FF)
                 *   3       %1x         Led Enabled State (0 = off, 1 = on)
                 *   4       %+05.01f    Board temperature in °C
                 *   5       %+05.01f    LED heatsink temperature in °C
                 *   6       %04d        Fan speed in rpm
                 *   7       %05.02f     Input voltage measurement
                 *   9       %04d        Front knob position as a percentage of full scale x 10 → (response of "0981" = 98.1%)
                 *   10      %04d        Analog input as a percentage of full scale (0-5V) x 10 → (response of "0981" = 98.1% → 5V*0.981=4.905V)
                 *   11      %1x         Status of front button (0 = not pressed, 1 = pressed)
                 *   12      %1x         Status of digital input (0 = low, 1 = high or disconnected)
                 *   13      %1x         Control source (0 = Front Knob, 1 = Analog Input, 2 = RS232 Port, 4 = USB Port, 7 = None, any other value is invalid)
                 */
                if (statusArray.Count() == 13)
                {
                    // warnings and faults and update the front panel indicators
                    Int32.TryParse(statusArray[0], System.Globalization.NumberStyles.HexNumber, null, out int faults);
                    Int32.TryParse(statusArray[1], System.Globalization.NumberStyles.HexNumber, null, out int warnings);
                    ProcessFaultIndicator(faults, warnings, ErrLedLabel, 0);                //bit mask = 1
                    ProcessFaultIndicator(faults, warnings, ErrFanLabel, 1);                //bit mask = 2
                    ProcessFaultIndicator(faults, warnings, ErrInputVoltageLabel, 2);       //bit mask = 4
                    ProcessFaultIndicator(faults, warnings, ErrLEDTemperatureLabel, 3);     //bit mask = 8
                    ProcessFaultIndicator(faults, warnings, ErrBoardTemperatureLabel, 4);  //bit mask = 16

                    // Update the intensity adjustment slider and associated text boxes, but not while the mouse is dragging it
                    if (!LedIntensitySlider.IsMouseCaptureWithin)
                        LedIntensitySlider.Value = Convert.ToInt32(statusArray[2], 16);
                    LedIntensityTextBoxes_Update();

                    // Update other controls
                    LedOutputEnableButton_Update(statusArray[3] == "1");
                    BoardTemperatureTextBox.Text = float.Parse(statusArray[4]).ToString();
                    HeatsinkTemperatureTextBox.Text = float.Parse(statusArray[5]).ToString();
                    FanSpeedTextBox.Text = statusArray[6];
                    InputVoltageTextBox.Text = float.Parse(statusArray[7]).ToString();
                    FrontAnalogInputSlider.Value = float.Parse(statusArray[8]) / 10.0;
                    RearAnalogInputSlider.Value = float.Parse(statusArray[9]) / 10.0;
                    FrontButtonCheckBox.IsChecked = statusArray[10] != "0";
                    DigitalInputCheckBox.IsChecked = statusArray[11] != "0";

                    // Determine which control source, and update the text box
                    string controlSource;
                    switch (Convert.ToInt32(statusArray[12]))
                    {
                        case 0: controlSource = "Front"; break;
                        case 1: controlSource = "Analog"; break;
                        case 2: controlSource = "RS232"; break;
                        case 4: controlSource = "USB"; break;
                        case 7: controlSource = "None"; break;
                        default: controlSource = String.Empty; break;
                    }
                    ControlSourceTextBox.Text = controlSource;
                }
            }
            
            // update remaining form controls not handled by &xs command
            FrontControlLockoutButton_Update();
            RearControlLockoutButton_Update();
            DigitalInputActiveStateButton_Update();
        }
        #endregion

        #region Control Output Functions
        private void LedOutputEnableButton_Click(object sender, RoutedEventArgs e)
        {
            var parameter = (bool)LedOutputEnableButton.IsChecked ? "1" : "0";
            SendCommand("&l", parameter);
            LedOutputEnableButton_Update(parameter == "1");
        }

        private void FrontControlLockoutButton_Click(object sender, RoutedEventArgs e)
        {
            var parameter = (bool)FrontControlLockoutButton.IsChecked ? "0" : "1";
            FrontControlLockoutButton_Update(parameter);
        }

        /// <summary>
        /// Updates the front control lockout button
        /// </summary>
        /// <param name="arg">The argument to send with the command, "?" if omitted.</param>
        private void FrontControlLockoutButton_Update(string arg = "?")
        {
            var response = SendCommand("&hlf", arg);
            switch (response)
            {
                case "0":
                    FrontControlLockoutButton.IsChecked = true;
                    FrontControlLockoutButton.Content = "Disabled";
                    break;
                case "1":
                    FrontControlLockoutButton.IsChecked = false;
                    FrontControlLockoutButton.Content = "Enabled";
                    break;
                default:
                    break;
            }
        }

        private void RearControlLockoutButton_Click(object sender, RoutedEventArgs e)
        {
            var parameter = (bool)RearControlLockoutButton.IsChecked ? "0" : "1";
            RearControlLockoutButton_Update(parameter);
        }

        /// <summary>
        /// Updates the rear control lockout button
        /// </summary>
        /// /// <param name="arg">The argument to send with the command, "?" if omitted.</param>
        private void RearControlLockoutButton_Update(string arg = "?")
        {
            var response = SendCommand("&hlm", arg);
            switch (response)
            {
                case "0":
                    RearControlLockoutButton.IsChecked = true;
                    RearControlLockoutButton.Content = "Disabled";
                    break;
                case "1":
                    RearControlLockoutButton.IsChecked = false;
                    RearControlLockoutButton.Content = "Enabled";
                    break;
                default:
                    break;
            }
        }

        private void DigitalInputActiveStateButton_Click(object sender, RoutedEventArgs e)
        {
            var parameter = (bool)DigitalInputActiveStateButton.IsChecked ? "1" : "0";
            DigitalInputActiveStateButton_Update(parameter);
        }

        /// <summary>
        /// Sends a command to either change or query the state of the digital input polarity (&j#)
        /// </summary>
        /// <param name="arg">The argument to send with the command, "?" if omitted.</param>
        private void DigitalInputActiveStateButton_Update(string arg = "?")
        {
            //var response = SendCommand("&j", "?");
            var response = SendCommand("&j", arg);
            switch (response)
            {
                case "0":
                    // 0 = low off (default), IsChecked=false
                    DigitalInputActiveStateButton.Content = "Normal";
                    DigitalInputActiveStateButton.IsChecked = false;
                    break;
                case "1":
                    // 1 = high off, IsChecked=true
                    DigitalInputActiveStateButton.Content = "Inverted";
                    DigitalInputActiveStateButton.IsChecked = true;
                    break;
            }
        }

        private void DigitalInputSwitchModeButton_Click(object sender, RoutedEventArgs e)
        {
            var parameter = (bool)DigitalInputSwitchModeButton.IsChecked ? "1" : "0";
            //SendCommand("&jm", parameter);
            DigitalInputSwitchModeButton_Update(parameter);
        }

        /// <summary>
        /// Query or change the state of the digital input switch mode (&jm#)
        /// </summary>
        /// <param name="arg">The argument to send with the command, "?" if omitted.</param>
        private void DigitalInputSwitchModeButton_Update(string arg = "?")
        {
            var response = SendCommand("&jm", arg);
            switch (response)
            {
                case "0":
                    DigitalInputSwitchModeButton.Content = "Level";
                    DigitalInputSwitchModeButton.IsChecked = false;
                    break;
                case "1":
                    DigitalInputSwitchModeButton.Content = "Edge";
                    DigitalInputSwitchModeButton.IsChecked = true;
                    break;
            }
        }

        /// <summary>
        /// Read the LED intensity value using the &ip command
        /// </summary>
        private void LedIntensityTextBoxes_Update()
        {
            LedIntensityHexTextBox.Text = $"0x{((int)(LedIntensitySlider.Value)).ToString("X3")}";
            LedIntensityDecTextBox.Text = $"{(LedIntensitySlider.Value / 20.47).ToString("0.0")}%";
        }

        private void LedIntensitySlider_ValueChanged(object sender, RoutedPropertyChangedEventArgs<double> e)
        {
            LedIntensityTextBoxes_Update();
            // The &IP command is used to adjust intensity, which takes a hex value between 0 and 7FF (decimal 2047)
            // The slider and associated textbox displays a value 0 to 2047.
            // To convert the slider value to the appropriate value for the &IP command,
            // the slider value must be converted to hex.

            // only send a command if the update was triggered by dragging the slider
            if (LedIntensitySlider.IsMouseCaptureWithin)
            {
                var parameter = $"{((int)LedIntensitySlider.Value).ToString("X")}";
                SendCommand("&ip", parameter, false);
            }
        }

        private void SaveUserSettingsButton_Click(object sender, RoutedEventArgs e) => SendCommand("&s");

        private void RestoreUserSettingsButton_Click(object sender, RoutedEventArgs e) => SendCommand("&t");

        private void RestoreFactorySettingsButton_Click(object sender, RoutedEventArgs e) => SendCommand("&o");

        private void ResetButton_Click(object sender, RoutedEventArgs e) => SendCommand("&o4");
        #endregion

        #region Form-update Functions
        /// <summary>
        /// Initialize the form to default values on startup
        /// </summary>
        private void InitializeForm()
        {
            UpdateComPortList();
            EnableFormControls(false);

            // Update title bar to show name and version of application
            var version = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version;
            this.Title = string.Format("MC-LS Demo - Version {0}.{1}",
                version.Major, version.Minor);

            // Configure tool tips to be displayed for the maximum amount of time.
            ToolTipService.ShowDurationProperty.OverrideMetadata(typeof(DependencyObject), new FrameworkPropertyMetadata(Int32.MaxValue));
        }
                
        /// <summary>
        /// Enable or disable certain form controls. The controls in this function are enabled/disabled when a connection to an MC-LS is made/lost.
        /// </summary>
        /// <param name="state">True = enabled, False = disabled</param>
        private void EnableFormControls(bool state = false)
        {
            // disable or enable certain controls
            var controlList = new List<Control>
            {
                DisconnectButton,
                LedOutputEnableButton,
                LedIntensitySlider,
                LedIntensityDecTextBox,
                LedIntensityHexTextBox,
                FrontControlLockoutButton,
                RearControlLockoutButton,
                DigitalInputActiveStateButton,
                DigitalInputSwitchModeButton,
                SaveUserSettingsButton,
                RestoreUserSettingsButton,
                RestoreFactorySettingsButton,
                ResetButton,
            };
            foreach (Control ctrl in controlList) ctrl.IsEnabled = state;
            IntensityControlGroup.IsEnabled = state;
            DigitalInputBehaviorButtons.IsEnabled = state;
            
            // reset text boxes
            if (state == false)
            {
                ModelNumberTextBox.Text =
                    SerialNumberTextBox.Text =
                    FirmwareTextBox.Text =
                    HeatsinkTemperatureTextBox.Text =
                    BoardTemperatureTextBox.Text =
                    InputVoltageTextBox.Text =
                    FanSpeedTextBox.Text =
                    ControlSourceTextBox.Text =
                    String.Empty;
                FrontAnalogInputTextBox.Text = RearAnalogInputTextBox.Text = "0.0";
                LedIntensityDecTextBox.Text = "0%";
                LedIntensityHexTextBox.Text = "0x000";
                FrontButtonCheckBox.IsChecked = DigitalInputCheckBox.IsChecked = false;
            }
        }

        private void ComPortListBox_DropDownOpened(object sender, EventArgs e) => UpdateComPortList();

        /// <summary>
        /// Update the COM Port combobox with a list of currently available COM ports.
        /// </summary>
        public void UpdateComPortList()
        {
            var comPortList = new List<string>();
            foreach (string s in SerialPort.GetPortNames())
            {
                if (!comPortList.Contains(s)) comPortList.Add(s);
            }
            comPortList.Sort();
            ComPortListBox.ItemsSource = comPortList;
        }

        /// <summary>
        /// Change the color of the error indicator label according to the type of fault.
        /// </summary>
        /// <param name="faults">The 'faults' bit-field returned by the &xs command</param>
        /// <param name="warnings">The 'warnings' bit-field returned by the &xs command</param>
        /// <param name="label">The name of the form Label to modify</param>
        /// <param name="mask">The index of the bit-field that should be used to format the label</param>
        private void ProcessFaultIndicator(int faults, int warnings, Label label, int bit)
        {
            int mask = (int)Math.Pow(2, bit);

            // fault → make control background red
            if ((faults & mask) != 0)
            {
                label.Background = Brushes.Red;
                label.Foreground = Brushes.Black;
                label.FontWeight = FontWeights.Bold; 
            }
            
            // warning → make control background yellow
            else if ((warnings & mask) != 0)
            {
                label.Background = Brushes.Yellow;
                label.Foreground = Brushes.Black;
                label.FontWeight = FontWeights.Bold;
            }

            // no warning or fault detected
            else
            {
                label.Background = Brushes.Green;
                label.Foreground = Brushes.White;
                label.FontWeight = FontWeights.Normal;
            }
            
        }

        /// <summary>
        /// Update the LedOutputButton state and button text
        /// </summary>
        /// <param name="state"></param>
        private void LedOutputEnableButton_Update(bool state)
        {
            switch (state)
            {
                case false:
                    LedOutputEnableButton.IsChecked = false;
                    LedOutputEnableButton.Content = "Off";
                    break;
                case true:
                    LedOutputEnableButton.IsChecked = true;
                    LedOutputEnableButton.Content = "On";
                    break;
            }
        }
        #endregion

        #region Serial Port Functions
        /// <summary>
        /// Close the serial port and update the UI
        /// </summary>
        private void SerialPort_Close()
        {
            // close com port
            if (null != serialPort)
            {
                try
                {
                    serialPort.Close();
                }
                catch (System.IO.IOException)
                {
                    // This error usually happens when the USB cable is removed before the .Close() method is called.
                }
                catch // any other exception
                {
                    tbErrorMessage.Text = "Error closing serial port.";
                }
                serialPort = null;
            }
        }

        private void DisconnectButton_Click(object sender, RoutedEventArgs e)
        {
            SerialPort_Disconnect();
        }

        /// <summary>
        /// Close the serial port and update the UI
        /// </summary>
        private void SerialPort_Disconnect()
        {
            SerialPort_Close();
            ComPortListBox.SelectedIndex = -1;
            EnableFormControls(false);
            lblConnected.Text = "Disconnected";
            lblConnected.Background = Brushes.Red;
            dispatcherTimer.Stop();
            tbErrorMessage.Text = "Select a COM port.";
        }
        
        /// <summary>
        /// Attempt to open the indicated COM port.
        /// </summary>
        /// <param name="selectedItem">The desired COM port (e.g. "COM1")</param>
        private void SerialPort_Connect(string selectedItem)
        {
            // close any open com ports
            SerialPort_Close();

            // try to open the selected serial port
            try
            {
                serialPort = new SerialPort(selectedItem, 9600)
                {
                    NewLine = "\r",
                    ReadTimeout = 250
                };
                serialPort.Open();
                lblConnected.Background = Brushes.LightGreen;
                lblConnected.Text = "Connected";
                EnableFormControls(true);

                // get the serial number
                SerialNumberTextBox.Text = SendCommand("&z");
                // get the model number
                ModelNumberTextBox.Text = SendCommand("&zm");
                // get the firmware revision
                FirmwareTextBox.Text = SendCommand("&f");
                // get the product identification string
                tbErrorMessage.Text = SendCommand("&q");

                // start polling the unit for status updates
                dispatcherTimer.Start();
            }
            catch (Exception ex)
            {
                string mbMessage;
                string mbCaption;

                if (ex is UnauthorizedAccessException)
                {
                    mbMessage = $"Another program is using {ComPortListBox.SelectedItem}.  Close the port in the other program and try again.";
                    mbCaption = "COM Port Not Accessible";
                }
                else if (ex is TimeoutException)
                {
                    mbMessage = $"An MC-LS was not found on {ComPortListBox.SelectedItem}.  Please try a different port.";
                    mbCaption = "MC-LS Not Found";
                }
                // Any other exception
                else
                {
                    mbMessage = $"Unable to access {ComPortListBox.SelectedItem}";
                    mbCaption = "COM Port Error";
                }
                
                SerialPort_Disconnect();
                ComPortListBox.SelectedIndex = -1;
                dispatcherTimer.Stop();
                MessageBox.Show(mbMessage, mbCaption, MessageBoxButton.OK, MessageBoxImage.Error);
            }
        }

        /// <summary>
        /// Try to open the selected COM port when it is selected from the combobox.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ComPortListBox_SelectionChanged(object sender, RoutedEventArgs e)
        {
            ComboBox item = (ComboBox)sender;

            // catch only when a port is selected, not when the combo box is initially clicked
            if (item.SelectedItem != null)
                SerialPort_Connect(item.SelectedItem.ToString());
        }

        /// <summary>
        /// Send a command to the MC-LS on the currently opened serialPort and process the response according to the arguments.
        /// The function prepends the start character "&" to the sent command so there is no need to include it in the command parameter.
        /// </summary>
        /// <param name="command">the characters to send immediately after the initial "&"</param>
        /// <param name="parameter">an additional string to send after the command string</param>
        /// <param name="removePrefix">true: remove a string equal in length to the "command" parameter from the front of the response. All MC-LS responses start with "&" followed by the command string, so this will be true for most cases.</param>
        /// <returns></returns>
        private string SendCommand(string command, string parameter = "", bool removePrefix = true)
        {
            try
            {
                serialPort.DiscardInBuffer();
                serialPort.WriteLine(command + parameter);
                var response = serialPort.ReadLine();
                return removePrefix ?
                    response.Remove(0, command.Length) :
                    response;
            }
            catch
            {
                return String.Empty;
            }
        }
        #endregion
    }
}
